(() => {
  // 設定で有効/無効を切り替え可能にする
  let enabled = null; // 未決定: 初期設定ロード後に start/stop
  let scanTimer = null;
  let mo = null;
  const attachedInputs = new Set();
  const handlers = new WeakMap();
  // ---- 1) まず「overage の見た目」を実測して複製CSSを作る ----
  function installOverageLookCSS(sampleLi) {
    try {
      const st = sampleLi.querySelector('.indicators .status');
      const sp = st?.querySelector('span');
      if (!st || !sp) return;

      const prev = sampleLi.dataset.status;
      sampleLi.dataset.status = 'overage'; // 一瞬だけ overage にして見た目を測る
      const csS = getComputedStyle(st), csP = getComputedStyle(sp);

      const pick = (cs, props) =>
        props.reduce((o, p) => {
          const v = cs.getPropertyValue(p);
          if (v && v !== 'initial' && v !== 'auto' && v !== 'normal') o[p] = v;
          return o;
        }, {});

      const sS = pick(csS, ['display','align-items','gap','color','background-color','border','padding','border-radius','box-shadow','margin']);
      const sP = pick(csP, ['display','color','font-size','font-weight','line-height']);
      sampleLi.dataset.status = prev;

      const decl = o => Object.entries(o).map(([k,v]) => `${k}:${v} !important;`).join('');
      const css = `
        /* 文字種不一致のとき "overage と同じ見た目" を出す */
        li.exercise-item.answer-area.type-descriptive.is-overage-look .indicators .status { ${decl(sS)} }
        li.exercise-item.answer-area.type-descriptive.is-overage-look .indicators .status span { ${decl(sP)} }
      `;

      let style = document.getElementById('overage-look-style');
      if (!style) {
        style = document.createElement('style');
        style.id = 'overage-look-style';
        document.head.appendChild(style);
      }
      style.textContent = css;
    } catch (e) {
      // Silent
    }
  }

  const sampleLi =
    document.querySelector('li.exercise-item.answer-area.type-descriptive') ||
    document.body.querySelector('li.exercise-item.answer-area.type-descriptive');
  if (sampleLi) installOverageLookCSS(sampleLi);

  // ---- 2) 文字数/文字種チェック（吹き出しは使わない） ----
  const inputsDone = new WeakSet();

  const toHalfDigits = s => s.replace(/[０-９]/g, ch => String.fromCharCode(ch.charCodeAt(0)-0xFF10+0x30));
  const cpLen = str => [...(str||'')].length; // 現状未使用だが保持

  const extractLimitFromQuestion = (txt) => {
    const t = toHalfDigits(txt||'');
    // より柔軟なパターンマッチング: 「7文字」「3文字」なども対応
    const m = t.match(/(\d+)\s*(?:文字|字)/);
    if (!m) return null;
    
    // 「程度」「以上」「前後」等の修飾語がある場合は無効にする
    const beforeLimit = t.substring(0, m.index);
    const afterLimit = t.substring(m.index + m[0].length);
    const modifierWords = /(程度|以上|前後)/;
    
    if (modifierWords.test(beforeLimit) || modifierWords.test(afterLimit)) {
      return null; // 修飾語がある場合は制限を適用しない
    }
    
    return parseInt(m[1], 10);
  };
  const detectCategory = (txt) => {
    if (!txt) return null;
    if (/(カタカナ)/.test(txt)) return 'katakana';
    if (/(ひらがな)/.test(txt)) return 'hiragana';
    if (/(漢字)/.test(txt)) return 'kanji';
    if (/(アルファベット|英字|ローマ字)/.test(txt)) return 'alphabet';
    if (/(数字|数値|半角数字)/.test(txt)) return 'digit';
    return null;
  };
  const parseWidthPrefs = (txt) => {
    const map={}; if(!txt) return map;
    const pairs = [
      {re: /(カタカナ)/, key:'katakana'},
      {re: /(ひらがな)/, key:'hiragana'},
      {re: /(漢字)/,     key:'kanji'},
      {re: /(アルファベット|英字|ローマ字)/, key:'alphabet'},
      {re: /(数字|数値)/, key:'digit'},
    ];
    for (const {re,key} of pairs) {
      const rHalf=new RegExp(re.source+'.*?(は)?\\s*半角'), rFull=new RegExp(re.source+'.*?(は)?\\s*全角');
      if (rHalf.test(txt)) map[key]='half';
      if (rFull.test(txt)) map[key]='full';
    }
    return map;
  };

  const Char = {
    isHiragana: c => /[\u3041-\u3096\u309D\u309E]/.test(c),
    isKatakanaFull: c => /[\u30A0-\u30FF]/.test(c),
    isKatakanaHalf: c => /[\uFF66-\uFF9F]/.test(c),
    isKanji: c => /[\u3400-\u9FFF\uF900-\uFAFF\u3005\u303B\u30F5\u30F6]/.test(c),
    isAlphabetHalf: c => /[A-Za-z]/.test(c),
    isAlphabetFull: c => /[\uFF21-\uFF3A\uFF41-\uFF5A]/.test(c),
    isDigitHalf: c => /[0-9]/.test(c),
    isDigitFull: c => /[\uFF10-\uFF19]/.test(c),
    isProlong: c => c==='ー',
    isMiddleDot: c => c==='・',
  };
  const charOK = (c, category, widthPref) => {
    switch (category) {
      case 'hiragana': if (widthPref==='half') return false; return Char.isHiragana(c);
      case 'katakana':
        if (widthPref==='half') return Char.isKatakanaHalf(c);
        if (widthPref==='full') return Char.isKatakanaFull(c)||Char.isProlong(c)||Char.isMiddleDot(c);
        return Char.isKatakanaFull(c)||Char.isKatakanaHalf(c)||Char.isProlong(c)||Char.isMiddleDot(c);
      case 'kanji': if (widthPref==='half') return false; return Char.isKanji(c);
      case 'alphabet':
        if (widthPref==='half') return Char.isAlphabetHalf(c);
        if (widthPref==='full') return Char.isAlphabetFull(c);
        return Char.isAlphabetHalf(c)||Char.isAlphabetFull(c);
      case 'digit':
        if (widthPref==='half') return Char.isDigitHalf(c);
        if (widthPref==='full') return Char.isDigitFull(c);
        return Char.isDigitHalf(c)||Char.isDigitFull(c);
      default: return true;
    }
  };
  const categoryLabel = k => ({hiragana:'ひらがな',katakana:'カタカナ',kanji:'漢字',alphabet:'アルファベット',digit:'数字'}[k]||'');
  const widthLabel = w => w==='half'?'（半角のみ）':w==='full'?'（全角のみ）':'';

  function attach(input){
    if (enabled === false) return;
    if (inputsDone.has(input)) return; inputsDone.add(input);
    const li = input.closest('li.exercise-item.answer-area.type-descriptive'); if (!li) return;
    // 遅延でCSS複製（初期取得に失敗した場合の保険）
    if (!document.getElementById('overage-look-style')) {
      installOverageLookCSS(li);
    }
    const update = () => {
      if (enabled === false) return; // 無効時は何もしない
      // タブ切替や動的変更に追従できるよう、その都度取得
      const section = li.closest('section.exercise');
      const qText = li.querySelector('.question')?.innerText || '';
      const limit = extractLimitFromQuestion(qText) ?? 30;
      const category = detectCategory(qText);
      const widthPrefs = parseWidthPrefs(section?.querySelector('.statement')?.innerText || '');
      const statusSpan = li.querySelector('.indicators .status span');
      const counterSpan = li.querySelector('.indicators .counter span');

      const val = input.value || '';
      const chars = [...val];
      const len = chars.length;
      if (counterSpan) counterSpan.textContent = `${len}文字`;

      const msgs = [];
      const over = len - limit;

      if (over > 0) msgs.push(`${over}文字オーバー`);

      if (category) {
        const widthPref = widthPrefs[category];
        const bad = [];
        for (const c of chars) if (!charOK(c, category, widthPref)) bad.push(c);
        if (bad.length) {
          const preview = bad.slice(0,3).join('');
          msgs.push(`${categoryLabel(category)}のみ可${widthLabel(widthPref)}：不一致 ${bad.length}文字（例:「${preview}」）`);
        }
      } else {
        for (const [key,pref] of Object.entries(widthPrefs)){
          if (pref==='half'){
            if (key==='digit'    && chars.some(Char.isDigitFull))    msgs.push('数字は半角指定：全角数字が含まれています');
            if (key==='alphabet' && chars.some(Char.isAlphabetFull)) msgs.push('アルファベットは半角指定：全角英字が含まれています');
            if (key==='katakana' && chars.some(Char.isKatakanaFull)) msgs.push('カタカナは半角指定：全角カタカナが含まれています');
          } else if (pref==='full'){
            if (key==='digit'    && chars.some(Char.isDigitHalf))    msgs.push('数字は全角指定：半角数字が含まれています');
            if (key==='alphabet' && chars.some(Char.isAlphabetHalf)) msgs.push('アルファベットは全角指定：半角英字が含まれています');
            if (key==='katakana' && chars.some(Char.isKatakanaHalf)) msgs.push('カタカナは全角指定：半角ｶﾀｶﾅが含まれています');
          }
        }
      }

      // ---- 表示ロジック（吹き出しは使わない） ----
      if (statusSpan) statusSpan.textContent = msgs.join('／') || '';

      if (over > 0) {
        // 本当に文字数超過のときはサイト既存の overage をそのまま使う
        li.classList.remove('is-overage-look');
        li.dataset.status = 'overage';
      } else if (msgs.length > 0) {
        // 文字種不一致など → data-status は現行ルールに従いつつ、見た目は overage に寄せる
        li.classList.add('is-overage-look');
        li.dataset.status = (len > 0 ? 'valid' : 'initial'); // サイトの状態管理は壊さない
      } else {
        li.classList.remove('is-overage-look');
        li.dataset.status = (len > 0 ? 'valid' : 'initial');
      }
    };

    input.addEventListener('input', update);
    input.addEventListener('change', update);
    handlers.set(input, update);
    attachedInputs.add(input);
    update();
  }

  // 初期化スキャン
  const scan = () => {
    if (enabled === false) return;
    try {
      if (!document.getElementById('overage-look-style')) {
        const li = document.querySelector('li.exercise-item.answer-area.type-descriptive');
        if (li) installOverageLookCSS(li);
      }
      document.querySelectorAll('section.exercise li.exercise-item.answer-area.type-descriptive input.answers').forEach(attach);
    } catch (e) {}
  };
  function start() {
    if (enabled === true) return;
    enabled = true;
    scan();
    const SCAN_INTERVAL_MS = 500;
    if (!scanTimer) scanTimer = setInterval(scan, SCAN_INTERVAL_MS);
    document.addEventListener('visibilitychange', visHandler);
    window.addEventListener('hashchange', scan);
    if (!mo) {
      mo = new MutationObserver(muts => {
        for (const m of muts) for (const n of m.addedNodes) {
          if (n.nodeType !== 1) continue;
          n.querySelectorAll?.('section.exercise li.exercise-item.answer-area.type-descriptive input.answers').forEach(attach);
          // .status が後から出現した場合にCSS適用を再試行
          if (!document.getElementById('overage-look-style')) {
            const li = (n.matches?.('li.exercise-item.answer-area.type-descriptive') ? n : n.querySelector?.('li.exercise-item.answer-area.type-descriptive')) || null;
            if (li) installOverageLookCSS(li);
          }
        }
      });
      try { mo.observe(document.documentElement, { childList:true, subtree:true }); } catch (e) {}
    }
  }

  function stop() {
    if (enabled === false) return;
    enabled = false;
    if (scanTimer) { clearInterval(scanTimer); scanTimer = null; }
    try { document.removeEventListener('visibilitychange', visHandler); } catch (e) {}
    try { window.removeEventListener('hashchange', scan); } catch (e) {}
    if (mo) { try { mo.disconnect(); } catch (e) {} mo = null; }
    // 既存のリスナー解除と軽いクリーンアップ
    attachedInputs.forEach(input => {
      const fn = handlers.get(input);
      if (fn) {
        try { input.removeEventListener('input', fn); } catch (e) {}
        try { input.removeEventListener('change', fn); } catch (e) {}
      }
      const li = input.closest('li.exercise-item.answer-area.type-descriptive');
      if (li) li.classList.remove('is-overage-look');
    });
    attachedInputs.clear();
  }

  function visHandler() {
    if (document.visibilityState === 'visible') scan();
  }

  // 設定初期読み込み
  try {
    chrome.storage.sync.get('enableCharlimitValidator', (res) => {
      const flag = res && Object.prototype.hasOwnProperty.call(res, 'enableCharlimitValidator') ? res.enableCharlimitValidator : true;
      if (flag === false) stop(); else start();
    });
  } catch (e) {
    // storage非対応環境: 有効化
    start();
  }

  // ポップアップからの変更通知
  try {
    chrome.runtime?.onMessage?.addListener((message) => {
      if (message?.type === 'settingChanged' && 'enableCharlimitValidator' in message) {
        if (message.enableCharlimitValidator) start(); else stop();
      }
    });
  } catch (e) {}

  // ストレージの直接変更を検知
  try {
    chrome.storage?.onChanged?.addListener((changes, area) => {
      if (area === 'sync' && changes.enableCharlimitValidator) {
        const nv = changes.enableCharlimitValidator.newValue;
        if (nv === false) stop(); else start();
      }
    });
  } catch (e) {}

})();
